import { GoogleGenAI, Chat } from "@google/genai";
import { ChatMessage } from "../types";

const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });

const MODEL_NAME = 'gemini-3-flash-preview';

export const createChatSession = (): Chat => {
  return ai.chats.create({
    model: MODEL_NAME,
    config: {
      systemInstruction: `You are an intelligent, friendly, and enthusiastic AI Tour Guide for a WeChat Mini-Program called "Smart Guide". 
      
      Your persona:
      - Helpful, polite, and slightly witty.
      - Knowledgeable about travel, history, culture, and photography.
      - You use emojis occasionally to keep the tone light 🌟.
      - Keep responses concise (under 150 words usually) unless asked for a detailed history.
      - If the user asks about a specific place, provide practical tips (best time to visit, ticket prices, hidden gems).
      
      Structure your answers with markdown for readability.`
    },
  });
};

export const sendMessageToAI = async (chat: Chat, message: string): Promise<string> => {
  try {
    const response = await chat.sendMessage({ message });
    return response.text || "I'm having trouble connecting to the travel network right now. Please try again!";
  } catch (error) {
    console.error("Gemini API Error:", error);
    return "Sorry, I encountered a temporary glitch. Let's try that again.";
  }
};

export const generateDestinationSummary = async (destinationName: string): Promise<string> => {
  try {
    const response = await ai.models.generateContent({
      model: MODEL_NAME,
      contents: `Write a catchy, 2-sentence marketing summary for a travel destination named "${destinationName}". It should make someone want to visit immediately.`,
    });
    return response.text || "A beautiful place waiting for you to explore.";
  } catch (error) {
    return "Explore the beauty of nature and culture.";
  }
};