import React, { useState } from 'react';
import { useLocation, useNavigate } from 'react-router-dom';
import { MapPin, Headphones, Camera, BookOpen, Star, ChevronRight } from 'lucide-react';
import { BackButton } from '../components/BackButton';
import { ScenicArea } from '../types';

const ScenicDetail: React.FC = () => {
  const location = useLocation();
  const navigate = useNavigate();
  const scenic = location.state?.scenic as ScenicArea;
  const [activeTab, setActiveTab] = useState<'intro' | 'poi' | 'travelogue' | 'album'>('intro');

  if (!scenic) {
    return <div className="p-8 text-center">未找到景点信息</div>;
  }

  // Mock Sub-data
  const pois = [
    { id: '101', name: '主要观景台', desc: '观看日出的最佳位置。', img: 'https://picsum.photos/200/200?random=20' },
    { id: '102', name: '古树名木', desc: '拥有千年历史的古老松树。', img: 'https://picsum.photos/200/200?random=21' },
    { id: '103', name: '历史碑刻', desc: '文人墨客留下的珍贵笔迹。', img: 'https://picsum.photos/200/200?random=22' },
  ];

  const travelogues = [
    { id: '201', title: '一次难忘的旅行', user: '小明', avatar: 'https://picsum.photos/50/50?random=1', likes: 120, img: 'https://picsum.photos/300/200?random=30' },
    { id: '202', title: '避坑指南', user: 'Traveler_A', avatar: 'https://picsum.photos/50/50?random=2', likes: 85, img: 'https://picsum.photos/300/200?random=31' },
  ];

  const album = [1, 2, 3, 4, 5, 6].map(i => `https://picsum.photos/400/400?random=${40 + i}`);

  return (
    <div className="bg-white min-h-screen pb-20">
      {/* Header Image Area */}
      <div className="relative h-64">
        <img src={scenic.coverImage} alt={scenic.name} className="w-full h-full object-cover" />
        <div className="absolute top-0 left-0 right-0 p-4 pt-safe flex justify-between items-center bg-gradient-to-b from-black/50 to-transparent">
          <BackButton color="text-white" />
          <div className="flex space-x-4">
             {/* Icons like share or favorite could go here */}
          </div>
        </div>
        <div className="absolute bottom-0 left-0 right-0 p-4 bg-gradient-to-t from-black/70 to-transparent">
          <h1 className="text-2xl font-bold text-white mb-1">{scenic.name}</h1>
          <div className="flex items-center text-white/90 text-xs">
            <MapPin size={12} className="mr-1" />
            {scenic.location}
          </div>
        </div>
      </div>

      {/* Quick Actions */}
      <div className="flex justify-around py-4 border-b border-gray-100">
        <button onClick={() => navigate('/guide')} className="flex flex-col items-center space-y-1">
          <div className="w-10 h-10 rounded-full bg-blue-50 flex items-center justify-center text-blue-500">
            <Headphones size={20} />
          </div>
          <span className="text-xs text-gray-600">AI导游</span>
        </button>
        <button onClick={() => setActiveTab('album')} className="flex flex-col items-center space-y-1">
          <div className="w-10 h-10 rounded-full bg-orange-50 flex items-center justify-center text-orange-500">
            <Camera size={20} />
          </div>
          <span className="text-xs text-gray-600">相册</span>
        </button>
        <button onClick={() => setActiveTab('travelogue')} className="flex flex-col items-center space-y-1">
          <div className="w-10 h-10 rounded-full bg-green-50 flex items-center justify-center text-green-500">
            <BookOpen size={20} />
          </div>
          <span className="text-xs text-gray-600">游记</span>
        </button>
      </div>

      {/* Tabs */}
      <div className="sticky top-0 bg-white z-10 flex border-b border-gray-100 shadow-sm">
        {['intro', 'poi', 'travelogue', 'album'].map((tab) => (
          <button
            key={tab}
            onClick={() => setActiveTab(tab as any)}
            className={`flex-1 py-3 text-sm font-medium transition-colors relative ${
              activeTab === tab ? 'text-[#07C160]' : 'text-gray-500'
            }`}
          >
            {tab === 'intro' && '介绍'}
            {tab === 'poi' && '景点'}
            {tab === 'travelogue' && '游记'}
            {tab === 'album' && '相册'}
            {activeTab === tab && (
              <div className="absolute bottom-0 left-1/2 transform -translate-x-1/2 w-4 h-0.5 bg-[#07C160] rounded-full" />
            )}
          </button>
        ))}
      </div>

      {/* Content */}
      <div className="p-4 min-h-[300px]">
        {activeTab === 'intro' && (
          <div className="space-y-4 animate-fade-in">
            <div className="bg-gray-50 p-4 rounded-xl">
              <h3 className="font-bold text-gray-900 mb-2">景区简介</h3>
              <p className="text-gray-600 text-sm leading-relaxed">
                {scenic.shortDesc} 这里是详细的介绍文本。利用AI生成更多关于{scenic.name}的历史文化背景、地质地貌特征以及游览注意事项。这是一个令人神往的地方，四季皆有不同的美景。
              </p>
            </div>
            <div>
              <h3 className="font-bold text-gray-900 mb-2">实用信息</h3>
              <div className="grid grid-cols-2 gap-3 text-sm">
                <div className="p-3 border rounded-lg bg-white">
                  <span className="block text-gray-400 text-xs">开放时间</span>
                  <span className="text-gray-800">08:00 - 17:30</span>
                </div>
                <div className="p-3 border rounded-lg bg-white">
                  <span className="block text-gray-400 text-xs">建议游玩</span>
                  <span className="text-gray-800">3-4 小时</span>
                </div>
              </div>
            </div>
          </div>
        )}

        {activeTab === 'poi' && (
          <div className="space-y-4 animate-fade-in">
            {pois.map((poi) => (
              <div key={poi.id} className="flex bg-white border border-gray-100 rounded-xl p-3 shadow-sm">
                <img src={poi.img} alt={poi.name} className="w-20 h-20 rounded-lg object-cover flex-shrink-0" />
                <div className="ml-3 flex flex-col justify-center">
                  <h4 className="font-bold text-gray-900">{poi.name}</h4>
                  <p className="text-xs text-gray-500 mt-1 line-clamp-2">{poi.desc}</p>
                </div>
                <div className="ml-auto flex items-center text-gray-300">
                    <ChevronRight size={16} />
                </div>
              </div>
            ))}
          </div>
        )}

        {activeTab === 'travelogue' && (
          <div className="space-y-4 animate-fade-in">
             {travelogues.map((log) => (
               <div key={log.id} className="bg-white border border-gray-100 rounded-xl overflow-hidden shadow-sm">
                 <img src={log.img} className="w-full h-32 object-cover" />
                 <div className="p-3">
                   <h4 className="font-bold text-sm mb-2">{log.title}</h4>
                   <div className="flex items-center justify-between">
                     <div className="flex items-center space-x-2">
                       <img src={log.avatar} className="w-5 h-5 rounded-full" />
                       <span className="text-xs text-gray-500">{log.user}</span>
                     </div>
                     <div className="flex items-center text-xs text-gray-400">
                        <Star size={10} className="mr-1" /> {log.likes}
                     </div>
                   </div>
                 </div>
               </div>
             ))}
          </div>
        )}

        {activeTab === 'album' && (
          <div className="grid grid-cols-2 gap-2 animate-fade-in">
            {album.map((url, idx) => (
              <img key={idx} src={url} alt="Gallery" className="w-full h-40 object-cover rounded-lg" />
            ))}
          </div>
        )}
      </div>
    </div>
  );
};

export default ScenicDetail;