import React, { useState } from 'react';
import { User as UserType } from '../types';
import { Settings, ChevronRight, Image, FileText, Heart, LogOut, Mic2, Languages, Volume2 } from 'lucide-react';
import { useNavigate } from 'react-router-dom';

interface ProfileProps {
  user: UserType;
  logout: () => void;
}

const Profile: React.FC<ProfileProps> = ({ user, logout }) => {
  const navigate = useNavigate();

  // Settings State (Mock)
  const [guideStyle, setGuideStyle] = useState('humorous');
  const [speechRate, setSpeechRate] = useState('normal');

  const handleLogout = () => {
      logout();
      navigate('/login');
  };

  return (
    <div className="min-h-screen bg-gray-50 pb-24">
      {/* Header */}
      <div className="bg-[#07C160] pt-12 pb-8 px-6 text-white relative overflow-hidden">
        <div className="absolute top-0 right-0 p-4 opacity-20">
             <Settings size={80} />
        </div>
        <div className="flex items-center space-x-4 relative z-10">
          <div className="w-16 h-16 rounded-full border-2 border-white/50 overflow-hidden bg-white">
            <img src={user.avatar} alt="Avatar" className="w-full h-full object-cover" />
          </div>
          <div>
            <h2 className="text-xl font-bold">{user.username}</h2>
            <p className="text-white/80 text-sm mt-1">ID: {user.id}</p>
          </div>
        </div>
      </div>

      {/* Stats */}
      <div className="flex bg-white py-4 mb-4 shadow-sm">
        <div className="flex-1 text-center border-r border-gray-100">
          <span className="block text-lg font-bold text-gray-800">12</span>
          <span className="text-xs text-gray-500">游记</span>
        </div>
        <div className="flex-1 text-center border-r border-gray-100">
          <span className="block text-lg font-bold text-gray-800">45</span>
          <span className="text-xs text-gray-500">收藏</span>
        </div>
        <div className="flex-1 text-center">
          <span className="block text-lg font-bold text-gray-800">128</span>
          <span className="text-xs text-gray-500">照片</span>
        </div>
      </div>

      {/* Guide Preferences */}
      <div className="mx-4 mb-2">
        <h3 className="text-xs font-bold text-gray-400 mb-2 uppercase tracking-wider">导游偏好设置</h3>
        <div className="bg-white rounded-xl overflow-hidden shadow-sm space-y-px">
           {/* Style */}
           <div className="flex items-center p-4 border-b border-gray-50 justify-between">
              <div className="flex items-center">
                  <Mic2 className="mr-3 text-[#07C160]" size={20} />
                  <span className="text-gray-700 text-sm font-medium">解说风格</span>
              </div>
              <div className="flex bg-gray-100 rounded-lg p-1">
                  <button 
                    onClick={() => setGuideStyle('humorous')}
                    className={`px-3 py-1 text-xs rounded-md transition-all ${guideStyle === 'humorous' ? 'bg-white shadow-sm text-[#07C160] font-bold' : 'text-gray-500'}`}
                  >
                    幽默风趣
                  </button>
                  <button 
                    onClick={() => setGuideStyle('professional')}
                    className={`px-3 py-1 text-xs rounded-md transition-all ${guideStyle === 'professional' ? 'bg-white shadow-sm text-[#07C160] font-bold' : 'text-gray-500'}`}
                  >
                    专业严谨
                  </button>
              </div>
           </div>

           {/* Rate */}
           <div className="flex items-center p-4 border-b border-gray-50 justify-between">
              <div className="flex items-center">
                  <Volume2 className="mr-3 text-[#07C160]" size={20} />
                  <span className="text-gray-700 text-sm font-medium">语速调节</span>
              </div>
              <select 
                value={speechRate}
                onChange={(e) => setSpeechRate(e.target.value)}
                className="bg-gray-100 text-gray-600 text-xs py-1.5 px-3 rounded-lg outline-none border-none"
              >
                  <option value="slow">较慢 (0.8x)</option>
                  <option value="normal">正常 (1.0x)</option>
                  <option value="fast">较快 (1.2x)</option>
              </select>
           </div>

           {/* Language */}
           <div className="flex items-center p-4 border-b border-gray-50 justify-between">
              <div className="flex items-center">
                  <Languages className="mr-3 text-[#07C160]" size={20} />
                  <span className="text-gray-700 text-sm font-medium">语言偏好</span>
              </div>
              <div className="flex items-center text-gray-500 text-xs">
                  <span>普通话</span>
                  <ChevronRight size={14} className="ml-1" />
              </div>
           </div>
        </div>
      </div>

      {/* Menu List */}
      <div className="bg-white mx-4 rounded-xl overflow-hidden shadow-sm space-y-px mt-4">
        {[
          { icon: FileText, label: '我的游记', color: 'text-blue-500' },
          { icon: Image, label: '我的相册', color: 'text-orange-500' },
          { icon: Heart, label: '我的收藏', color: 'text-red-500' },
        ].map((item, idx) => (
          <div key={idx} className="flex items-center p-4 active:bg-gray-50 transition-colors cursor-pointer border-b border-gray-50 last:border-0">
            <item.icon className={`mr-3 ${item.color}`} size={20} />
            <span className="flex-1 text-gray-700 text-sm font-medium">{item.label}</span>
            <ChevronRight size={16} className="text-gray-300" />
          </div>
        ))}
      </div>

      <div className="bg-white mx-4 rounded-xl overflow-hidden shadow-sm space-y-px mt-4">
        <div className="flex items-center p-4 active:bg-gray-50 transition-colors cursor-pointer border-b border-gray-50">
             <Settings className="mr-3 text-gray-500" size={20} />
             <span className="flex-1 text-gray-700 text-sm font-medium">通用设置</span>
             <ChevronRight size={16} className="text-gray-300" />
        </div>
      </div>

       <div className="mx-4 mt-6">
           <button 
             onClick={handleLogout}
             className="w-full bg-white text-red-500 font-medium py-3 rounded-xl shadow-sm flex items-center justify-center"
            >
               <LogOut size={18} className="mr-2" />
               退出登录
           </button>
       </div>
    </div>
  );
};

export default Profile;