import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { Search, MapPin, CloudSun, ChevronRight, Play, Clock, Utensils, Ticket, Image as ImageIcon } from 'lucide-react';
import { ScenicArea } from '../types';

// Enhanced Mock Data
const CURRENT_SPOT: ScenicArea = {
  id: '2',
  name: '西湖风景名胜区',
  shortDesc: '欲把西湖比西子，淡妆浓抹总相宜。',
  fullDesc: '西湖，位于浙江省杭州市西湖区，是中国大陆首批国家重点风景名胜区。',
  coverImage: 'https://picsum.photos/800/600?random=2',
  location: '浙江省杭州市',
  tags: ['5A景区', '世界遗产', '免费'],
  pois: [
    { id: 'p1', name: '断桥残雪', description: '许仙白娘子相会之地。', imageUrl: 'https://picsum.photos/300/300?random=101', rating: 4.8 },
    { id: 'p2', name: '雷峰夕照', description: '位于西湖南岸夕照山上。', imageUrl: 'https://picsum.photos/300/300?random=102', rating: 4.7 },
    { id: 'p3', name: '三潭印月', description: '人民币一元纸币背景。', imageUrl: 'https://picsum.photos/300/300?random=103', rating: 4.9 },
  ],
  performances: [
    { id: 's1', name: '印象西湖', time: '19:45 - 20:45', location: '岳湖景区', imageUrl: 'https://picsum.photos/300/300?random=201' },
    { id: 's2', name: '西湖音乐喷泉', time: '19:00 / 20:00', location: '湖滨三公园', imageUrl: 'https://picsum.photos/300/300?random=202' },
  ],
  foods: [
    { id: 'f1', name: '西湖醋鱼', description: '酸甜鲜嫩，传统名菜', imageUrl: 'https://picsum.photos/300/300?random=301', rating: 4.2 },
    { id: 'f2', name: '龙井虾仁', description: '茶香浓郁，虾仁滑嫩', imageUrl: 'https://picsum.photos/300/300?random=302', rating: 4.8 },
    { id: 'f3', name: '定胜糕', description: '松软香糯，传统小吃', imageUrl: 'https://picsum.photos/300/300?random=303', rating: 4.5 },
  ],
  travelogues: [
    { 
      id: 't1', 
      type: 'video',
      userId: 'u1', 
      userName: '旅行Vlog酱', 
      userAvatar: 'https://picsum.photos/50/50?random=u1', 
      title: '西湖音乐喷泉现场实拍！太震撼了✨', 
      content: '现场氛围感拉满，一定要早点去占位置！', 
      images: ['https://picsum.photos/400/300?random=t1'], 
      likes: 1245, 
      date: '10分钟前' 
    },
    { 
      id: 't2', 
      type: 'album',
      userId: 'u2', 
      userName: '摄影师阿伟', 
      userAvatar: 'https://picsum.photos/50/50?random=u2', 
      title: '三月西湖，桃红柳绿醉人心', 
      content: '虽然人很多，但是早起真的很美。', 
      images: ['https://picsum.photos/400/300?random=t2', 'https://picsum.photos/400/300?random=t2b'],
      likes: 856,
      date: '1小时前'
    },
  ],
  album: ['https://picsum.photos/400/300?random=a1', 'https://picsum.photos/400/300?random=a2']
};

const Home: React.FC = () => {
  const navigate = useNavigate();
  const [searchQuery, setSearchQuery] = useState('');

  const handleScenicClick = () => {
    navigate(`/scenic/${CURRENT_SPOT.id}`, { state: { scenic: CURRENT_SPOT } });
  };

  return (
    <div className="min-h-screen bg-gray-50 pb-24">
      {/* Header / Search */}
      <div className="bg-white p-4 pt-safe sticky top-0 z-20 shadow-sm">
        <div className="flex items-center space-x-2">
           <div className="flex-1 bg-gray-100 rounded-full flex items-center px-4 py-2">
              <Search size={18} className="text-gray-400" />
              <input 
                type="text" 
                placeholder="搜索景点/美食/攻略" 
                className="bg-transparent border-none outline-none ml-2 text-sm w-full"
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
              />
           </div>
           <div className="flex flex-col items-center">
              <div className="text-xs font-bold text-gray-800 flex items-center">
                 <MapPin size={12} className="mr-0.5 text-[#07C160]" />
                 杭州
              </div>
              <div className="text-[10px] text-gray-500 flex items-center mt-0.5">
                 <CloudSun size={10} className="mr-1" />
                 24°C
              </div>
           </div>
        </div>
      </div>

      {/* Featured Banner */}
      <div className="p-4">
        <div className="relative rounded-2xl overflow-hidden shadow-md h-48" onClick={handleScenicClick}>
           <img src={CURRENT_SPOT.coverImage} className="w-full h-full object-cover" alt="Featured" />
           <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent flex flex-col justify-end p-4">
              <span className="bg-[#07C160] text-white text-[10px] px-2 py-0.5 rounded-md w-fit mb-2">今日推荐</span>
              <h2 className="text-white font-bold text-lg">{CURRENT_SPOT.name}</h2>
              <p className="text-white/80 text-xs mt-1 line-clamp-1">{CURRENT_SPOT.shortDesc}</p>
           </div>
        </div>
      </div>

      {/* Categories */}
      <div className="grid grid-cols-4 gap-4 px-4 mb-6">
         {[
           { icon: Ticket, color: 'text-blue-500', bg: 'bg-blue-50', name: '景点门票' },
           { icon: Utensils, color: 'text-orange-500', bg: 'bg-orange-50', name: '特色美食' },
           { icon: ImageIcon, color: 'text-purple-500', bg: 'bg-purple-50', name: 'AI修图' },
           { icon: Clock, color: 'text-green-500', bg: 'bg-green-50', name: '行程规划' },
         ].map((item, i) => (
           <div key={i} className="flex flex-col items-center space-y-2">
              <div className={`w-12 h-12 rounded-full ${item.bg} flex items-center justify-center ${item.color}`}>
                 <item.icon size={24} />
              </div>
              <span className="text-xs text-gray-600 font-medium">{item.name}</span>
           </div>
         ))}
      </div>

      {/* Travelogue Feed */}
      <div className="px-4">
         <div className="flex items-center justify-between mb-3">
            <h3 className="font-bold text-gray-900 text-lg">热门游记</h3>
            <div className="flex items-center text-xs text-gray-400">
               查看更多 <ChevronRight size={14} />
            </div>
         </div>

         <div className="masonry-grid space-y-4">
             {/* Left Column simulated for list */}
             {CURRENT_SPOT.travelogues.map((item) => (
               <div key={item.id} className="bg-white rounded-xl overflow-hidden shadow-sm">
                  <div className="relative">
                    <img src={item.images[0]} className="w-full h-48 object-cover" alt="" />
                    {item.type === 'video' && (
                       <div className="absolute top-2 right-2 w-6 h-6 bg-black/30 backdrop-blur-sm rounded-full flex items-center justify-center">
                          <Play size={12} className="text-white ml-0.5" />
                       </div>
                    )}
                  </div>
                  <div className="p-3">
                     <p className="text-sm font-bold text-gray-800 line-clamp-2 mb-2">{item.title}</p>
                     <div className="flex items-center justify-between">
                        <div className="flex items-center space-x-1.5">
                           <img src={item.userAvatar} className="w-5 h-5 rounded-full" alt="" />
                           <span className="text-xs text-gray-500">{item.userName}</span>
                        </div>
                        <span className="text-xs text-gray-400">{item.likes}赞</span>
                     </div>
                  </div>
               </div>
             ))}
         </div>
      </div>
    </div>
  );
};

export default Home;