import React, { useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { MessageCircle, Smartphone, CheckCircle2, Circle, ChevronLeft, X } from 'lucide-react';
import { AppRoute } from '../types';

interface AuthProps {
  setIsLoggedIn: (value: boolean) => void;
}

const Auth: React.FC<AuthProps> = ({ setIsLoggedIn }) => {
  const navigate = useNavigate();
  const [loading, setLoading] = useState(false);
  const [agreed, setAgreed] = useState(false);
  const [showPhoneLogin, setShowPhoneLogin] = useState(false);

  // Phone login states
  const [phone, setPhone] = useState('');
  const [code, setCode] = useState('');
  const [countdown, setCountdown] = useState(0);

  const handleLoginSuccess = () => {
    setLoading(true);
    // Simulate API call
    setTimeout(() => {
      setLoading(false);
      setIsLoggedIn(true);
      navigate(AppRoute.HOME);
    }, 800);
  };

  const handleWechatLogin = () => {
    if (!agreed) {
        // Shake animation or toast could go here
        alert('请先阅读并同意《用户协议》和《隐私政策》');
        return;
    }
    handleLoginSuccess();
  };

  const handlePhoneLogin = (e: React.FormEvent) => {
    e.preventDefault();
    if (!agreed) {
        alert('请先阅读并同意《用户协议》和《隐私政策》');
        return;
    }
    if (!phone || !code) return;
    handleLoginSuccess();
  };

  const startCountdown = () => {
      if (countdown > 0 || !phone) return;
      setCountdown(60);
      const timer = setInterval(() => {
          setCountdown(prev => {
              if (prev <= 1) {
                  clearInterval(timer);
                  return 0;
              }
              return prev - 1;
          });
      }, 1000);
  };

  // Phone Login View
  if (showPhoneLogin) {
      return (
          <div className="min-h-screen bg-white px-6 pt-12 flex flex-col">
              <button onClick={() => setShowPhoneLogin(false)} className="mb-8 text-gray-800">
                  <X size={24} />
              </button>
              
              <h1 className="text-2xl font-bold text-gray-900 mb-2">手机号登录</h1>
              <p className="text-gray-500 text-sm mb-8">未注册的手机号验证通过后将自动注册</p>

              <form onSubmit={handlePhoneLogin} className="space-y-6">
                  <div className="border-b border-gray-200 py-3">
                      <input 
                        type="tel" 
                        placeholder="请输入手机号"
                        className="w-full outline-none text-lg placeholder-gray-300"
                        value={phone}
                        onChange={(e) => setPhone(e.target.value)}
                      />
                  </div>
                  <div className="border-b border-gray-200 py-3 flex justify-between items-center">
                      <input 
                        type="number" 
                        placeholder="请输入验证码"
                        className="w-full outline-none text-lg placeholder-gray-300"
                        value={code}
                        onChange={(e) => setCode(e.target.value)}
                      />
                      <button 
                        type="button"
                        onClick={startCountdown}
                        disabled={countdown > 0 || !phone}
                        className={`text-sm font-medium ${countdown > 0 || !phone ? 'text-gray-300' : 'text-[#07C160]'}`}
                      >
                          {countdown > 0 ? `${countdown}s后重发` : '获取验证码'}
                      </button>
                  </div>

                  <button
                    type="submit"
                    disabled={loading}
                    className="w-full py-3.5 rounded-lg bg-[#07C160] text-white font-bold text-lg shadow-sm disabled:opacity-70 mt-8 active:scale-[0.98] transition-transform"
                  >
                    {loading ? '登录中...' : '登 录'}
                  </button>
              </form>

              <div className="mt-auto mb-8 flex items-start space-x-2">
                <button onClick={() => setAgreed(!agreed)} className="mt-0.5">
                    {agreed ? <CheckCircle2 size={16} className="text-[#07C160]" /> : <Circle size={16} className="text-gray-300" />}
                </button>
                <p className="text-xs text-gray-400 leading-tight">
                    我已阅读并同意
                    <span className="text-blue-600">《用户协议》</span>
                    及
                    <span className="text-blue-600">《隐私政策》</span>
                </p>
              </div>
          </div>
      );
  }

  // Default "WeChat" Style Login View
  return (
    <div className="min-h-screen bg-white flex flex-col items-center px-8 pt-32 pb-12 relative overflow-hidden">
      {/* Background Decor */}
      <div className="absolute top-[-10%] right-[-10%] w-64 h-64 bg-green-50 rounded-full blur-3xl opacity-60"></div>
      <div className="absolute bottom-[-10%] left-[-10%] w-64 h-64 bg-blue-50 rounded-full blur-3xl opacity-60"></div>

      {/* App Branding */}
      <div className="flex flex-col items-center mb-24 z-10 animate-fade-in">
        <div className="w-20 h-20 bg-gradient-to-br from-[#07C160] to-[#05a050] rounded-2xl shadow-xl flex items-center justify-center mb-4 transform rotate-3">
            <span className="text-white text-3xl font-bold">AI</span>
        </div>
        <h1 className="text-2xl font-bold text-gray-900 tracking-wide">AI 智能导游</h1>
        <p className="text-sm text-gray-500 mt-2">您的贴身旅行管家</p>
      </div>

      {/* Action Buttons */}
      <div className="w-full space-y-4 z-10">
        <button
          onClick={handleWechatLogin}
          disabled={loading}
          className="w-full py-3.5 rounded-full bg-[#07C160] text-white font-medium text-base shadow-lg shadow-green-100 flex items-center justify-center space-x-2 active:scale-[0.98] transition-transform"
        >
          <MessageCircle size={20} fill="currentColor" className="text-white" />
          <span>微信一键登录</span>
        </button>
        
        <button
          onClick={() => setShowPhoneLogin(true)}
          className="w-full py-3.5 rounded-full bg-gray-50 text-gray-700 font-medium text-base flex items-center justify-center space-x-2 hover:bg-gray-100 active:scale-[0.98] transition-transform"
        >
          <Smartphone size={20} />
          <span>手机号码登录</span>
        </button>
      </div>

      {/* Agreement Footer */}
      <div className="mt-auto flex items-start space-x-2 max-w-[280px]">
        <button onClick={() => setAgreed(!agreed)} className="mt-0.5 flex-shrink-0">
            {agreed ? <CheckCircle2 size={16} className="text-[#07C160]" /> : <Circle size={16} className="text-gray-300" />}
        </button>
        <p className="text-xs text-gray-400 leading-tight text-center">
            登录即代表您已同意
            <span className="text-blue-600">《用户协议》</span>
            和
            <span className="text-blue-600">《隐私政策》</span>
            ，并授权使用您的账号信息（如昵称、头像、地区及性别）
        </p>
      </div>
    </div>
  );
};

export default Auth;