import React, { useState, useEffect, useRef } from 'react';
import { Mic, MicOff, PhoneOff, Video, VideoOff, Sparkles, Loader2 } from 'lucide-react';
import { GoogleGenAI, LiveServerMessage, Modality } from '@google/genai';

// --- Audio Helper Functions ---
function encode(bytes: Uint8Array) {
  let binary = '';
  const len = bytes.byteLength;
  for (let i = 0; i < len; i++) {
    binary += String.fromCharCode(bytes[i]);
  }
  return btoa(binary);
}

function decode(base64: string) {
  const binaryString = atob(base64);
  const len = binaryString.length;
  const bytes = new Uint8Array(len);
  for (let i = 0; i < len; i++) {
    bytes[i] = binaryString.charCodeAt(i);
  }
  return bytes;
}

async function decodeAudioData(
  data: Uint8Array,
  ctx: AudioContext,
  sampleRate: number,
  numChannels: number,
): Promise<AudioBuffer> {
  const dataInt16 = new Int16Array(data.buffer);
  const frameCount = dataInt16.length / numChannels;
  const buffer = ctx.createBuffer(numChannels, frameCount, sampleRate);

  for (let channel = 0; channel < numChannels; channel++) {
    const channelData = buffer.getChannelData(channel);
    for (let i = 0; i < frameCount; i++) {
      channelData[i] = dataInt16[i * numChannels + channel] / 32768.0;
    }
  }
  return buffer;
}

function createBlob(data: Float32Array): { data: string; mimeType: string } {
  const l = data.length;
  const int16 = new Int16Array(l);
  for (let i = 0; i < l; i++) {
    int16[i] = data[i] * 32768;
  }
  return {
    data: encode(new Uint8Array(int16.buffer)),
    mimeType: 'audio/pcm;rate=16000',
  };
}

async function blobToBase64(blob: Blob): Promise<string> {
  return new Promise((resolve, reject) => {
    const reader = new FileReader();
    reader.onloadend = () => {
      const base64String = (reader.result as string).split(',')[1];
      resolve(base64String);
    };
    reader.onerror = reject;
    reader.readAsDataURL(blob);
  });
}

const AIGuide: React.FC = () => {
  const [status, setStatus] = useState<'idle' | 'connecting' | 'connected' | 'error'>('idle');
  const [isMuted, setIsMuted] = useState(false);
  const [isVideoOn, setIsVideoOn] = useState(true);
  
  const videoRef = useRef<HTMLVideoElement>(null);
  const canvasRef = useRef<HTMLCanvasElement>(null);
  const visualizerRef = useRef<HTMLCanvasElement>(null);
  
  // Audio Refs
  const inputAudioContextRef = useRef<AudioContext | null>(null);
  const outputAudioContextRef = useRef<AudioContext | null>(null);
  const analyserRef = useRef<AnalyserNode | null>(null);
  
  // Logic Refs
  const nextStartTimeRef = useRef<number>(0);
  const sourcesRef = useRef<Set<AudioBufferSourceNode>>(new Set());
  const frameIntervalRef = useRef<number | null>(null);
  const sessionRef = useRef<Promise<any> | null>(null);

  const startSession = async () => {
    try {
      setStatus('connecting');

      // 1. Setup Audio Contexts
      const inputCtx = new (window.AudioContext || (window as any).webkitAudioContext)({ sampleRate: 16000 });
      const outputCtx = new (window.AudioContext || (window as any).webkitAudioContext)({ sampleRate: 24000 });
      inputAudioContextRef.current = inputCtx;
      outputAudioContextRef.current = outputCtx;

      // 2. Setup User Media (Camera + Mic)
      const stream = await navigator.mediaDevices.getUserMedia({ audio: true, video: true });
      if (videoRef.current) {
        videoRef.current.srcObject = stream;
        videoRef.current.play();
      }

      // 3. Audio Visualization Setup
      const analyser = inputCtx.createAnalyser();
      analyser.fftSize = 256;
      analyserRef.current = analyser;
      const source = inputCtx.createMediaStreamSource(stream);
      const scriptProcessor = inputCtx.createScriptProcessor(4096, 1, 1);
      
      source.connect(analyser); // Connect mic to analyser for visualization
      source.connect(scriptProcessor);
      scriptProcessor.connect(inputCtx.destination);

      // 4. Connect to Gemini Live
      const ai = new GoogleGenAI({ apiKey: process.env.API_KEY });
      const sessionPromise = ai.live.connect({
        model: 'gemini-2.5-flash-native-audio-preview-09-2025',
        config: {
          responseModalities: [Modality.AUDIO],
          speechConfig: {
            voiceConfig: { prebuiltVoiceConfig: { voiceName: 'Kore' } },
          },
          systemInstruction: 'You are an energetic and helpful tour guide looking through the user\'s camera. Describe what you see, answer questions about the scenery, and be concise.',
        },
        callbacks: {
          onopen: () => {
            console.log('Gemini Live Connected');
            setStatus('connected');
            
            // Start Audio Streaming
            scriptProcessor.onaudioprocess = (e) => {
              if (isMuted) return;
              const inputData = e.inputBuffer.getChannelData(0);
              const pcmBlob = createBlob(inputData);
              sessionPromise.then(session => session.sendRealtimeInput({ media: pcmBlob }));
            };

            // Start Video Streaming
            startVideoStreaming(sessionPromise);
          },
          onmessage: async (message: LiveServerMessage) => {
             // Handle Audio Output
             const base64Audio = message.serverContent?.modelTurn?.parts[0]?.inlineData?.data;
             if (base64Audio) {
               const ctx = outputAudioContextRef.current;
               if (ctx) {
                 nextStartTimeRef.current = Math.max(nextStartTimeRef.current, ctx.currentTime);
                 const audioBuffer = await decodeAudioData(decode(base64Audio), ctx, 24000, 1);
                 const source = ctx.createBufferSource();
                 source.buffer = audioBuffer;
                 source.connect(ctx.destination);
                 source.start(nextStartTimeRef.current);
                 nextStartTimeRef.current += audioBuffer.duration;
                 sourcesRef.current.add(source);
                 source.onended = () => sourcesRef.current.delete(source);
               }
             }
             // Handle Interruption
             if (message.serverContent?.interrupted) {
               sourcesRef.current.forEach(s => s.stop());
               sourcesRef.current.clear();
               nextStartTimeRef.current = 0;
             }
          },
          onclose: () => {
            console.log('Gemini Live Closed');
            setStatus('idle');
          },
          onerror: (err) => {
            console.error('Gemini Live Error', err);
            setStatus('error');
          }
        }
      });
      
      sessionRef.current = sessionPromise;

    } catch (error) {
      console.error("Failed to start session:", error);
      setStatus('error');
    }
  };

  const startVideoStreaming = (sessionPromise: Promise<any>) => {
    if (frameIntervalRef.current) clearInterval(frameIntervalRef.current);
    
    frameIntervalRef.current = window.setInterval(() => {
      if (!videoRef.current || !canvasRef.current || !isVideoOn) return;
      
      const video = videoRef.current;
      const canvas = canvasRef.current;
      const ctx = canvas.getContext('2d');
      if (!ctx) return;

      canvas.width = video.videoWidth * 0.5; // Scale down for performance
      canvas.height = video.videoHeight * 0.5;
      ctx.drawImage(video, 0, 0, canvas.width, canvas.height);
      
      canvas.toBlob(async (blob) => {
        if (blob) {
          const base64 = await blobToBase64(blob);
          sessionPromise.then(session => {
             session.sendRealtimeInput({
               media: { data: base64, mimeType: 'image/jpeg' }
             });
          });
        }
      }, 'image/jpeg', 0.6);
    }, 1000); // Send 1 frame per second
  };

  // Visualizer Loop
  useEffect(() => {
    if (status !== 'connected') return;
    
    const draw = () => {
      if (!analyserRef.current || !visualizerRef.current) {
        requestAnimationFrame(draw);
        return;
      }
      
      const canvas = visualizerRef.current;
      const ctx = canvas.getContext('2d');
      if (!ctx) return;

      const dataArray = new Uint8Array(analyserRef.current.frequencyBinCount);
      analyserRef.current.getByteFrequencyData(dataArray);

      // Simple Wave Visualizer
      ctx.clearRect(0, 0, canvas.width, canvas.height);
      const centerX = canvas.width / 2;
      const centerY = canvas.height / 2;
      const radius = 30; // Base radius

      // Calculate average volume
      let sum = 0;
      for(let i = 0; i < dataArray.length; i++) sum += dataArray[i];
      const average = sum / dataArray.length;
      
      const dynamicRadius = radius + (average * 0.5);

      ctx.beginPath();
      ctx.arc(centerX, centerY, dynamicRadius, 0, 2 * Math.PI);
      ctx.fillStyle = 'rgba(7, 193, 96, 0.6)'; // WeChat Green
      ctx.fill();
      
      // Outer ring
      ctx.beginPath();
      ctx.arc(centerX, centerY, dynamicRadius + 10, 0, 2 * Math.PI);
      ctx.strokeStyle = 'rgba(255, 255, 255, 0.8)';
      ctx.lineWidth = 2;
      ctx.stroke();

      requestAnimationFrame(draw);
    };
    draw();
  }, [status]);

  useEffect(() => {
    startSession();
    return () => {
      // Cleanup
      if (frameIntervalRef.current) clearInterval(frameIntervalRef.current);
      if (sessionRef.current) {
          // sessionRef.current.then(s => s.close()); // close not always available on type, handle gracefully
      }
      inputAudioContextRef.current?.close();
      outputAudioContextRef.current?.close();
      const stream = videoRef.current?.srcObject as MediaStream;
      stream?.getTracks().forEach(t => t.stop());
    };
  }, []);

  return (
    <div className="relative h-screen bg-black overflow-hidden flex flex-col">
      {/* Hidden Canvas for Frame Capture */}
      <canvas ref={canvasRef} className="hidden" />

      {/* Video Feed */}
      <video 
        ref={videoRef} 
        className="absolute inset-0 w-full h-full object-cover" 
        playsInline 
        muted // Mute local echo
      />

      {/* Overlays */}
      <div className="relative z-10 flex-1 flex flex-col justify-between p-6 pb-24 safe-area-pb">
        
        {/* Header Status */}
        <div className="flex justify-between items-start pt-safe">
          <div className="flex items-center space-x-2 bg-black/40 backdrop-blur-md px-3 py-1.5 rounded-full border border-white/10">
             <Sparkles className="text-[#07C160]" size={16} />
             <span className="text-white text-xs font-bold">AI 导游</span>
             {status === 'connecting' && <span className="text-white/60 text-xs flex items-center"><Loader2 size={10} className="animate-spin ml-1"/> 连接中...</span>}
             {status === 'connected' && <span className="text-[#07C160] text-xs flex items-center">● 在线</span>}
             {status === 'error' && <span className="text-red-400 text-xs">连接断开</span>}
          </div>
        </div>

        {/* Center Visualizer Area */}
        <div className="flex-1 flex items-center justify-center relative">
           {/* Visualizer Canvas */}
           <canvas ref={visualizerRef} width={300} height={300} className="w-64 h-64" />
           {status === 'connected' && (
             <div className="absolute text-white/80 text-sm font-medium mt-32 text-center">
                我正在看... <br/>
                <span className="text-xs opacity-60">请说话或展示周围环境</span>
             </div>
           )}
        </div>

        {/* Bottom Controls */}
        <div className="flex items-center justify-center space-x-8">
          <button 
             onClick={() => {
                 const newVal = !isVideoOn;
                 setIsVideoOn(newVal);
                 // In a real app we might toggle the track enabled state
             }}
             className={`p-4 rounded-full backdrop-blur-md transition-all ${isVideoOn ? 'bg-white/20 text-white' : 'bg-white text-gray-900'}`}
          >
             {isVideoOn ? <Video size={24} /> : <VideoOff size={24} />}
          </button>

          <button className="w-20 h-20 bg-red-500 rounded-full flex items-center justify-center shadow-lg active:scale-95 transition-transform">
             <PhoneOff size={32} className="text-white" />
          </button>

          <button 
             onClick={() => setIsMuted(!isMuted)}
             className={`p-4 rounded-full backdrop-blur-md transition-all ${!isMuted ? 'bg-white/20 text-white' : 'bg-white text-gray-900'}`}
          >
             {isMuted ? <MicOff size={24} /> : <Mic size={24} />}
          </button>
        </div>
      </div>
    </div>
  );
};

export default AIGuide;