import React from 'react';
import { useNavigate, useLocation } from 'react-router-dom';
import { Home, Map, User, Compass } from 'lucide-react';
import { AppRoute } from '../types';

export const TabBar: React.FC = () => {
  const navigate = useNavigate();
  const location = useLocation();

  const isActive = (path: string) => {
    // Handle root path specifically or check if pathname starts with the route
    if (path === '/') return location.pathname === '/';
    return location.pathname.startsWith(path);
  };

  const tabs = [
    { name: '首页', path: AppRoute.HOME, icon: Home },
    { name: 'AI导游', path: AppRoute.GUIDE, icon: Compass },
    { name: '我的', path: AppRoute.PROFILE, icon: User },
  ];

  // Hide tab bar on login/register pages
  if (location.pathname === AppRoute.LOGIN || location.pathname === AppRoute.REGISTER) {
    return null;
  }

  // Also hide on detail pages to give full screen immersive feel (optional, but common in apps)
  // For this demo, we'll keep it on main tabs only.
  const showTabs = [AppRoute.HOME, AppRoute.GUIDE, AppRoute.PROFILE].includes(location.pathname as AppRoute);
  
  // Actually, let's keep it simple: Show on top level pages, hide on details
  const isTopLevel = location.pathname === '/' || location.pathname === '/guide' || location.pathname === '/profile';

  if (!isTopLevel) return null;

  return (
    <div className="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 pb-safe pt-2 px-6 flex justify-between items-center z-50 h-16 safe-area-pb">
      {tabs.map((tab) => {
        const Icon = tab.icon;
        const active = isActive(tab.path);
        return (
          <button
            key={tab.path}
            onClick={() => navigate(tab.path)}
            className={`flex flex-col items-center justify-center w-full space-y-1 ${
              active ? 'text-[#07C160]' : 'text-gray-400'
            }`}
          >
            <Icon size={24} strokeWidth={active ? 2.5 : 2} />
            <span className="text-[10px] font-medium">{tab.name}</span>
          </button>
        );
      })}
    </div>
  );
};