import React, { useState, useEffect } from 'react';
import { HashRouter, Routes, Route, Navigate, useLocation } from 'react-router-dom';
import { TabBar } from './components/TabBar';
import Home from './pages/Home';
import Auth from './pages/Auth';
import AIGuide from './pages/AIGuide';
import Profile from './pages/Profile';
import ScenicDetail from './pages/ScenicDetail';
import { User, AppRoute } from './types';

// Wrapper to handle auth redirection
const ProtectedRoute = ({ children, isLoggedIn }: React.PropsWithChildren<{ isLoggedIn: boolean }>) => {
  if (!isLoggedIn) {
    return <Navigate to={AppRoute.LOGIN} replace />;
  }
  return <>{children}</>;
};

const ScrollToTop = () => {
    const { pathname } = useLocation();
    useEffect(() => {
        window.scrollTo(0, 0);
    }, [pathname]);
    return null;
}

const App: React.FC = () => {
  // Simple auth state persistence
  const [isLoggedIn, setIsLoggedIn] = useState<boolean>(() => {
      return localStorage.getItem('isLoggedIn') === 'true';
  });

  const [user, setUser] = useState<User>({
    id: '8888',
    username: '旅行者',
    avatar: 'https://picsum.photos/200/200?random=99',
    isLoggedIn: false
  });

  useEffect(() => {
      localStorage.setItem('isLoggedIn', String(isLoggedIn));
  }, [isLoggedIn]);

  const handleLogout = () => {
      setIsLoggedIn(false);
  };

  return (
    <HashRouter>
      <ScrollToTop />
      <div className="max-w-md mx-auto min-h-screen bg-gray-50 shadow-2xl overflow-hidden relative" id="app-container">
        <Routes>
          {/* Public Routes */}
          <Route path={AppRoute.LOGIN} element={<Auth setIsLoggedIn={setIsLoggedIn} />} />
          <Route path={AppRoute.REGISTER} element={<Auth setIsLoggedIn={setIsLoggedIn} />} />

          {/* Protected Routes */}
          <Route path={AppRoute.HOME} element={
            <ProtectedRoute isLoggedIn={isLoggedIn}>
              <Home />
            </ProtectedRoute>
          } />
          
          <Route path={AppRoute.GUIDE} element={
            <ProtectedRoute isLoggedIn={isLoggedIn}>
              <AIGuide />
            </ProtectedRoute>
          } />

          <Route path={AppRoute.PROFILE} element={
            <ProtectedRoute isLoggedIn={isLoggedIn}>
              <Profile user={user} logout={handleLogout} />
            </ProtectedRoute>
          } />

          <Route path="/scenic/:id" element={
            <ProtectedRoute isLoggedIn={isLoggedIn}>
              <ScenicDetail />
            </ProtectedRoute>
          } />

          {/* Catch all */}
          <Route path="*" element={<Navigate to={AppRoute.HOME} replace />} />
        </Routes>

        <TabBar />
      </div>
    </HashRouter>
  );
};

export default App;